-- SQL schema for Futbol Torneos v1
CREATE TABLE IF NOT EXISTS users (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(100) NOT NULL,
  email VARCHAR(150) NOT NULL UNIQUE,
  password VARCHAR(255) NOT NULL,
  role ENUM('admin','organizer','referee','manager') NOT NULL DEFAULT 'admin',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS tournaments (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(120) NOT NULL,
  season VARCHAR(20) NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS teams (
  id INT AUTO_INCREMENT PRIMARY KEY,
  name VARCHAR(120) NOT NULL,
  short_name VARCHAR(10),
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS matches (
  id INT AUTO_INCREMENT PRIMARY KEY,
  round INT NOT NULL,
  home_team_id INT NOT NULL,
  away_team_id INT NOT NULL,
  home_goals INT DEFAULT NULL,
  away_goals INT DEFAULT NULL,
  played_at DATETIME DEFAULT NULL,
  stage ENUM('league','semi','final') NOT NULL DEFAULT 'league',
  leg TINYINT NOT NULL DEFAULT 1,
  CONSTRAINT fk_home FOREIGN KEY (home_team_id) REFERENCES teams(id) ON DELETE CASCADE,
  CONSTRAINT fk_away FOREIGN KEY (away_team_id) REFERENCES teams(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Seed admin (password: admin123)
INSERT INTO users (name,email,password,role) VALUES
('Administrador','admin@example.com', '$2y$10$1wKQyY7WZfZ9s0Nfsf7r3ejzWf9o9p1z6r5w9qYt7O3mG7pS5la2W', 'admin');
-- The above hash is a bcrypt for 'admin123'

-- v3 additions
ALTER TABLE tournaments
  ADD COLUMN knockout_target ENUM('R16','QF') NOT NULL DEFAULT 'QF',
  ADD COLUMN direct_qualifiers INT NOT NULL DEFAULT 4,
  ADD COLUMN playin_slots INT NOT NULL DEFAULT 0,
  ADD COLUMN two_legs TINYINT NOT NULL DEFAULT 0;

ALTER TABLE matches
  ADD COLUMN tournament_id INT NULL,
  ADD CONSTRAINT fk_match_tournament FOREIGN KEY (tournament_id) REFERENCES tournaments(id) ON DELETE CASCADE;

CREATE TABLE IF NOT EXISTS tournament_teams (
  id INT AUTO_INCREMENT PRIMARY KEY,
  tournament_id INT NOT NULL,
  team_id INT NOT NULL,
  UNIQUE KEY uniq_tt (tournament_id, team_id),
  CONSTRAINT fk_tt_t FOREIGN KEY (tournament_id) REFERENCES tournaments(id) ON DELETE CASCADE,
  CONSTRAINT fk_tt_team FOREIGN KEY (team_id) REFERENCES teams(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- v4 additions
CREATE TABLE IF NOT EXISTS players (
  id INT AUTO_INCREMENT PRIMARY KEY,
  team_id INT NOT NULL,
  number INT DEFAULT NULL,
  name VARCHAR(120) NOT NULL,
  position ENUM('GK','DF','MF','FW') DEFAULT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_players_team FOREIGN KEY (team_id) REFERENCES teams(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS match_lineups (
  id INT AUTO_INCREMENT PRIMARY KEY,
  match_id INT NOT NULL,
  team_id INT NOT NULL,
  player_id INT NOT NULL,
  is_starter TINYINT NOT NULL DEFAULT 0,
  minute_in INT DEFAULT NULL,
  minute_out INT DEFAULT NULL,
  CONSTRAINT fk_ml_match FOREIGN KEY (match_id) REFERENCES matches(id) ON DELETE CASCADE,
  CONSTRAINT fk_ml_team FOREIGN KEY (team_id) REFERENCES teams(id) ON DELETE CASCADE,
  CONSTRAINT fk_ml_player FOREIGN KEY (player_id) REFERENCES players(id) ON DELETE CASCADE,
  UNIQUE KEY uniq_lineup (match_id, player_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS match_events (
  id INT AUTO_INCREMENT PRIMARY KEY,
  match_id INT NOT NULL,
  team_id INT NOT NULL,
  player_id INT DEFAULT NULL,
  minute INT NOT NULL,
  type ENUM('goal','own_goal','penalty_goal','yellow','red','sub_in','sub_out') NOT NULL,
  related_player_id INT DEFAULT NULL,
  notes VARCHAR(255) DEFAULT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_me_match FOREIGN KEY (match_id) REFERENCES matches(id) ON DELETE CASCADE,
  CONSTRAINT fk_me_team FOREIGN KEY (team_id) REFERENCES teams(id) ON DELETE CASCADE,
  CONSTRAINT fk_me_player FOREIGN KEY (player_id) REFERENCES players(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- v5 additions
CREATE TABLE IF NOT EXISTS player_suspensions (
  id INT AUTO_INCREMENT PRIMARY KEY,
  player_id INT NOT NULL,
  team_id INT NOT NULL,
  tournament_id INT NOT NULL,
  origin_match_id INT NOT NULL,
  match_id INT DEFAULT NULL, -- match where suspension applies
  matches_remaining INT NOT NULL DEFAULT 1,
  reason ENUM('red','double_yellow') NOT NULL,
  active TINYINT NOT NULL DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_ps_player FOREIGN KEY (player_id) REFERENCES players(id) ON DELETE CASCADE,
  CONSTRAINT fk_ps_team FOREIGN KEY (team_id) REFERENCES teams(id) ON DELETE CASCADE,
  CONSTRAINT fk_ps_tournament FOREIGN KEY (tournament_id) REFERENCES tournaments(id) ON DELETE CASCADE,
  CONSTRAINT fk_ps_origin FOREIGN KEY (origin_match_id) REFERENCES matches(id) ON DELETE CASCADE,
  CONSTRAINT fk_ps_match FOREIGN KEY (match_id) REFERENCES matches(id) ON DELETE SET NULL,
  UNIQUE KEY uniq_player_origin (player_id, origin_match_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- v6 additions
CREATE TABLE IF NOT EXISTS sanction_rules (
  id INT AUTO_INCREMENT PRIMARY KEY,
  tournament_id INT NOT NULL UNIQUE,
  red_min_matches INT NOT NULL DEFAULT 1,
  double_yellow_matches INT NOT NULL DEFAULT 1,
  yellow_threshold INT NOT NULL DEFAULT 5,
  yellow_threshold_matches INT NOT NULL DEFAULT 1,
  yellow_reset_on_suspension TINYINT NOT NULL DEFAULT 1,
  carry_over_to_playoffs TINYINT NOT NULL DEFAULT 1,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_sr_t FOREIGN KEY (tournament_id) REFERENCES tournaments(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

CREATE TABLE IF NOT EXISTS player_card_accum (
  id INT AUTO_INCREMENT PRIMARY KEY,
  player_id INT NOT NULL,
  team_id INT NOT NULL,
  tournament_id INT NOT NULL,
  yellows INT NOT NULL DEFAULT 0,
  last_match_id INT DEFAULT NULL,
  UNIQUE KEY uniq_accum (player_id, tournament_id),
  CONSTRAINT fk_pca_player FOREIGN KEY (player_id) REFERENCES players(id) ON DELETE CASCADE,
  CONSTRAINT fk_pca_t FOREIGN KEY (tournament_id) REFERENCES tournaments(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- For existing installs:
-- ALTER TABLE player_suspensions MODIFY COLUMN reason ENUM('red','double_yellow','accum_yellows') NOT NULL;
