<?php
namespace App\Models;

class Playoff {
    public static function getByStage($tournament_id, $stage) {
        $pdo = Database::conn();
        $sql = "SELECT m.*, h.name AS home_name, a.name AS away_name
                FROM matches m
                LEFT JOIN teams h ON h.id = m.home_team_id
                LEFT JOIN teams a ON a.id = m.away_team_id
                WHERE m.stage=? AND m.tournament_id=? ORDER BY id";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$stage, $tournament_id]);
        return $stmt->fetchAll();
    }
    public static function getFinal($tournament_id) {
        $pdo = Database::conn();
        $sql = "SELECT m.*, h.name AS home_name, a.name AS away_name
                FROM matches m
                LEFT JOIN teams h ON h.id = m.home_team_id
                LEFT JOIN teams a ON a.id = m.away_team_id
                WHERE stage='final' AND tournament_id=? LIMIT 1";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$tournament_id]);
        $res = $stmt->fetchAll();
        return $res ? $res[0] : null;
    }

    public static function generateAccordingToRules(array $table, array $rules, int $tournament_id) {
        $pdo = Database::conn();
        $target = $rules['knockout_target']; // 'R16' or 'QF'
        $direct = (int)$rules['direct_qualifiers'];
        $playin_slots = (int)$rules['playin_slots'];
        $target_size = ($target === 'R16') ? 16 : 8;

        // Clean existing KO for this tournament
        $stmt = $pdo->prepare("DELETE FROM matches WHERE tournament_id=? AND stage IN ('playin','R16','QF','semi','final')");
        $stmt->execute([$tournament_id]);

        // Determine main bracket and play-in ranges
        $remaining = max(0, $target_size - $direct);
        if ($playin_slots !== $remaining) {
            // align play-in slots to remaining
            $playin_slots = $remaining;
        }

        // Seeds from standings
        $seeds = array_column($table, 'team_id');

        // Create play-in (pairs among next 2*playin_slots teams)
        $qualified = array_slice($seeds, 0, $direct);
        $cursor = $direct;
        $playin_winners_placeholder = [];
        if ($playin_slots > 0) {
            $pool = array_slice($seeds, $cursor, 2*$playin_slots);
            // pair highest vs lowest within pool
            for ($i=0; $i<$playin_slots; $i++) {
                $home = $pool[$i];
                $away = $pool[count($pool)-1-$i];
                MatchModel::createWithStage($home, $away, 1, 'playin', 1, $tournament_id);
                // placeholder winner slot
                $playin_winners_placeholder[] = ['home'=>$home,'away'=>$away];
            }
            $cursor += 2*$playin_slots;
        }

        // Assemble main bracket participants
        // Winners of play-in fill lowest seeds; qualified remain top seeds
        $participants = $qualified;
        // we will reference play-in winners as 0; later refresh will set real teams
        for ($i=0; $i<$playin_slots; $i++) $participants[] = 0;

        // Create main bracket round (R16 or QF)
        $stage_code = $target === 'R16' ? 'R16' : 'QF';
        $n = $target_size;
        for ($i=0; $i<$n/2; $i++) {
            $home = $participants[$i] ?? 0;
            $away = $participants[$n-1-$i] ?? 0;
            MatchModel::createWithStage($home, $away, 1, $stage_code, 1, $tournament_id);
        }

        // Generate subsequent rounds placeholders
        // Semis (if target QF -> semis; if target R16 -> QF + semis)
        if ($target === 'R16') {
            // QF placeholders
            for ($i=0; $i<4; $i++) {
                MatchModel::createWithStage(0, 0, 2, 'QF', 1, $tournament_id);
            }
        }
        // Semis placeholders
        for ($i=0; $i<2; $i++) {
            MatchModel::createWithStage(0, 0, 3, 'semi', 1, $tournament_id);
        }
        // Final placeholder
        MatchModel::createWithStage(0, 0, 4, 'final', 1, $tournament_id);
    }

    public static function refreshProgression($tournament_id) {
        $pdo = Database::conn();
        // Resolve play-in to fill R16/QF empty teams (id=0 placeholders)
        $playins = self::getByStage($tournament_id, 'playin');
        $winners = [];
        foreach ($playins as $m) {
            if ($m['home_goals'] === null || $m['away_goals'] === null) continue;
            $winners[] = ($m['home_goals'] > $m['away_goals']) ? $m['home_team_id'] : ($m['away_goals'] > $m['home_goals'] ? $m['away_team_id'] : $m['home_team_id']);
        }
        if (!empty($winners)) {
            // fill zeros in R16/QF in reverse order
            $stages = ['R16','QF'];
            foreach ($stages as $st) {
                $stmt = $pdo->prepare("SELECT id, home_team_id, away_team_id FROM matches WHERE tournament_id=? AND stage=? ORDER BY id");
                $stmt->execute([$tournament_id, $st]);
                $rows = $stmt->fetchAll();
                for ($i=count($rows)-1; $i>=0 && !empty($winners); $i--) {
                    $row = $rows[$i];
                    if ((int)$row['away_team_id']===0) {
                        $tid = array_pop($winners);
                        $pdo->prepare("UPDATE matches SET away_team_id=? WHERE id=?")->execute([$tid, $row['id']]);
                    } elseif ((int)$row['home_team_id']===0) {
                        $tid = array_pop($winners);
                        $pdo->prepare("UPDATE matches SET home_team_id=? WHERE id=?")->execute([$tid, $row['id']]);
                    }
                }
            }
        }

        // Advance winners to next rounds
        self::advanceStage($tournament_id, 'R16', 'QF');
        self::advanceStage($tournament_id, 'QF', 'semi');
        self::advanceStage($tournament_id, 'semi', 'final');
    }

    private static function advanceStage($tournament_id, $from, $to) {
        $pdo = Database::conn();
        $stmt = $pdo->prepare("SELECT * FROM matches WHERE tournament_id=? AND stage=? ORDER BY id");
        $stmt->execute([$tournament_id, $from]);
        $fromMatches = $stmt->fetchAll();
        if (empty($fromMatches)) return;
        // Ensure 'to' matches exist
        $stmt2 = $pdo->prepare("SELECT id, home_team_id, away_team_id FROM matches WHERE tournament_id=? AND stage=? ORDER BY id");
        $stmt2->execute([$tournament_id, $to]);
        $toMatches = $stmt2->fetchAll();
        if (empty($toMatches)) return;

        $winners = [];
        foreach ($fromMatches as $m) {
            if ($m['home_goals'] === null || $m['away_goals'] === null) return;
            $winners[] = ($m['home_goals'] > $m['away_goals']) ? $m['home_team_id'] : ($m['away_goals'] > $m['home_goals'] ? $m['away_team_id'] : $m['home_team_id']);
        }
        // pair winners sequentially into next stage
        $idx = 0;
        foreach ($toMatches as $tm) {
            if ($idx+1 >= count($winners)) break;
            $h = $winners[$idx++];
            $a = $winners[$idx++];
            $pdo->prepare("UPDATE matches SET home_team_id=?, away_team_id=? WHERE id=?")->execute([$h,$a,$tm['id']]);
        }
    }
}
